"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebhookServiceModule = exports.WebhookService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const api_service_1 = require("../api.service");
const webhook_entity_1 = require("../../entities/webhook.entity");
const webhook_enum_1 = require("../../enums/webhook.enum");
const sms_enum_1 = require("../../enums/sms/sms.enum");
const webhook_repository_1 = require("../../repositories/webhook.repository");
const utility_service_1 = require("../utility.service");
const la_nest_library_1 = require("@serene-dev/la-nest-library");
const sms_repository_1 = require("../../repositories/sms/sms.repository");
const partner_repository_1 = require("../../repositories/partner.repository");
const sms_transaction_repository_1 = require("../../repositories/sms/sms-transaction.repository");
let WebhookService = class WebhookService {
    constructor(apiService, webhookRepository, smsRepository, smsTransactionRepository, partnerRepository) {
        this.apiService = apiService;
        this.webhookRepository = webhookRepository;
        this.smsRepository = smsRepository;
        this.smsTransactionRepository = smsTransactionRepository;
        this.partnerRepository = partnerRepository;
        this.maxRetries = 3;
        this.retryDelay = 5000;
    }
    async sendSMSStatusUpdateWebhook(conf, partner, config) {
        try {
            const eventType = this.getWebhookEventType(conf.smsTransaction.status);
            if (!partner.webhookUrl) {
                const msg = `Partner has no webhook URL configured`;
                la_nest_library_1.logger.info({
                    partnerId: partner.id,
                    smsTransactionId: conf.smsTransaction.id,
                }, msg);
                if (config?.throwError)
                    utility_service_1.UtilityClass.throwError({ message: msg });
                return;
            }
            const payload = this.buildWebhookPayload(conf, eventType);
            const webhook = await this.webhookRepository.save(this.webhookRepository.create({
                partnerId: partner.id,
                smsTransactionId: conf.smsTransaction.id,
                url: partner.webhookUrl,
                payload,
                status: webhook_enum_1.EWebhookStatus.pending,
                event: eventType,
                retryCount: 0,
            }));
            setImmediate(() => {
                this.sendWebhook(webhook).catch((error) => {
                    la_nest_library_1.logger.error({
                        webhookId: webhook.id,
                        error: error.message,
                    }, 'Failed to send webhook');
                });
            });
            la_nest_library_1.logger.info({
                webhookId: webhook.id,
                partnerId: partner.id,
                smsTransactionId: conf.smsTransaction.id,
                eventType,
            }, 'Webhook notification queued');
        }
        catch (error) {
            la_nest_library_1.logger.error({
                smsTransactionId: conf.smsTransaction.id,
                partnerId: partner.id,
                error: error.message,
            }, 'Failed to create webhook notification');
            if (config?.throwError)
                utility_service_1.UtilityClass.throwError(error);
        }
    }
    getWebhookEventType(status) {
        switch (status) {
            case sms_enum_1.ESMSStatus.sent:
                return webhook_enum_1.EWebhookEventType.smsSent;
            case sms_enum_1.ESMSStatus.delivered:
                return webhook_enum_1.EWebhookEventType.smsDelivered;
            case sms_enum_1.ESMSStatus.failed:
                return webhook_enum_1.EWebhookEventType.smsFailed;
            default:
                return webhook_enum_1.EWebhookEventType.smsStatusUpdated;
        }
    }
    async sendWebhookBySMSTransactionId(smsId, auth) {
        const transaction = await this.smsTransactionRepository.findOne({
            where: {
                id: smsId,
                recipient: { sms: { clientId: auth.isAdmin ? null : auth.orgId } },
            },
            relations: { recipient: { sms: { shortCode: true } } },
        });
        if (!transaction)
            utility_service_1.UtilityClass.throwError({
                message: `SMS doesn't exist`,
                statusCode: 404,
            });
        const orgId = transaction.recipient.sms.clientId;
        if (!orgId)
            utility_service_1.UtilityClass.throwError({
                message: `Client could not be found`,
                statusCode: 404,
            });
        const partner = await this.partnerRepository.findOneBy({
            id: orgId,
        });
        if (!partner)
            utility_service_1.UtilityClass.throwError({
                message: `Client could not be found`,
                statusCode: 404,
            });
        return await this.sendSMSStatusUpdateWebhook({
            smsTransaction: transaction,
            smsRecipient: transaction.recipient,
            sms: transaction.recipient.sms,
            shortCode: transaction.recipient.sms.shortCode,
        }, partner, {
            throwError: true,
        });
    }
    buildWebhookPayload(conf, eventType) {
        return {
            event: eventType,
            timestamp: new Date().toISOString(),
            data: {
                shortCode: conf.shortCode?.code || '',
                status: conf.smsTransaction.status,
                error: conf.smsTransaction.error,
                errorCode: conf.smsTransaction.pErrorCode || '',
                errorMessage: conf.smsTransaction.pErrorMessage || '',
                amount: conf.smsTransaction.amount,
                dateTimeSent: conf.smsTransaction.dateTimeSent,
                dateTimeDelivered: conf.smsTransaction.dateTimeDelivered,
                dateTimeFailed: conf.smsTransaction.dateTimeFailed,
                recipient: conf.smsRecipient?.recipient || '',
                message: conf.sms?.message || '',
                transactionId: conf.smsTransaction.id,
                reference: conf.smsTransaction.id,
                messageId: conf.sms.id,
                clientId: conf.sms.clientId || '',
            },
        };
    }
    async sendWebhook(webhook) {
        try {
            const response = await this.apiService.post(webhook.url, webhook.payload, {
                axiosConfig: {
                    timeout: 10000,
                    headers: {
                        'Content-Type': 'application/json',
                        'User-Agent': 'Bulk-SMS-Webhook/1.0',
                    },
                },
            });
            await this.webhookRepository.update(webhook.id, {
                status: webhook_enum_1.EWebhookStatus.delivered,
                responseStatus: response?.status || 200,
                responseBody: JSON.stringify(response?.data),
                deliveredAt: Date.now(),
            });
            la_nest_library_1.logger.info({
                webhookId: webhook.id,
                partnerId: webhook.partnerId,
                status: response?.status,
            }, 'Webhook delivered successfully');
        }
        catch (error) {
            await this.handleWebhookError(webhook, error);
        }
    }
    async handleWebhookError(webhook, error) {
        const newRetryCount = webhook.retryCount + 1;
        const errorMessage = error?.response?.data?.message || error?.message || 'Unknown error';
        if (newRetryCount <= this.maxRetries) {
            await this.webhookRepository.update({ id: webhook.id }, {
                status: webhook_enum_1.EWebhookStatus.retrying,
                retryCount: newRetryCount,
                errorMessage,
                responseStatus: error?.response?.status,
                responseBody: error?.response?.data
                    ? JSON.stringify(error.response.data)
                    : null,
            });
            setTimeout(() => {
                this.webhookRepository
                    .findOne({ where: { id: webhook.id } })
                    .then((updatedWebhook) => {
                    if (updatedWebhook) {
                        this.sendWebhook(updatedWebhook).catch((retryError) => {
                            la_nest_library_1.logger.error({
                                webhookId: webhook.id,
                                retryCount: newRetryCount,
                                error: retryError.message,
                            }, 'Webhook retry failed');
                        });
                    }
                });
            }, this.retryDelay * newRetryCount);
            la_nest_library_1.logger.info({
                webhookId: webhook.id,
                retryCount: newRetryCount,
                nextRetryIn: this.retryDelay * newRetryCount,
            }, 'Webhook retry scheduled');
        }
        else {
            await this.webhookRepository.update({ id: webhook.id }, {
                status: webhook_enum_1.EWebhookStatus.failed,
                retryCount: newRetryCount,
                errorMessage,
                responseStatus: error?.response?.status,
                responseBody: error?.response?.data
                    ? JSON.stringify(error.response.data)
                    : null,
            });
            la_nest_library_1.logger.error({
                webhookId: webhook.id,
                retryCount: newRetryCount,
                error: errorMessage,
            }, 'Webhook failed after max retries');
        }
    }
    async getWebhookStats(partnerId) {
        const stats = await this.webhookRepository
            .createQueryBuilder('webhook')
            .select('webhook.status', 'status')
            .addSelect('COUNT(*)', 'count')
            .where('webhook.partnerId = :partnerId', { partnerId })
            .groupBy('webhook.status')
            .getRawMany();
        const result = {
            total: 0,
            delivered: 0,
            failed: 0,
            pending: 0,
            retrying: 0,
        };
        stats.forEach((stat) => {
            const count = parseInt(stat.count);
            result.total += count;
            result[stat.status.toLowerCase()] = count;
        });
        return result;
    }
    async search({ errorMessage, responseBody, ...query }, auth) {
        if (auth.isClient || auth.isPublic)
            query.partnerId = auth.id;
        if (!query.pageNumber)
            query.pageNumber = 1;
        if (!query.pageSize)
            query.pageSize = 10;
        return utility_service_1.UtilityClass.search(this.webhookRepository, query, {
            baseWhere: {
                errorMessage: utility_service_1.UtilityClass.likeFormatter(errorMessage),
                responseBody: utility_service_1.UtilityClass.likeFormatter(responseBody),
            },
            baseRelations: { partner: !query.partnerId },
        });
    }
    async resendWebhook(webhookId, auth) {
        try {
            const webhook = await this.webhookRepository.findOne({
                where: { id: webhookId },
                relations: { partner: true, smsTransaction: true },
            });
            if (!webhook) {
                utility_service_1.UtilityClass.throwError({
                    message: 'Webhook not found',
                    statusCode: 404,
                });
            }
            if (auth.isClient && webhook.partnerId !== auth.id) {
                utility_service_1.UtilityClass.throwError({
                    message: 'Unauthorized to resend this webhook',
                    statusCode: 403,
                });
            }
            if (webhook.status === webhook_enum_1.EWebhookStatus.delivered) {
                utility_service_1.UtilityClass.throwError({
                    message: 'Webhook already delivered successfully',
                    statusCode: 400,
                });
            }
            await this.webhookRepository.update({ id: webhookId }, {
                status: webhook_enum_1.EWebhookStatus.pending,
                errorMessage: null,
                responseStatus: null,
                responseBody: null,
                deliveredAt: null,
            });
            const updatedWebhook = await this.webhookRepository.findOne({
                where: { id: webhookId },
            });
            if (updatedWebhook) {
                setImmediate(() => {
                    this.sendWebhook(updatedWebhook).catch((error) => {
                        la_nest_library_1.logger.error({
                            webhookId,
                            error: error.message,
                        }, 'Failed to resend webhook');
                    });
                });
            }
            la_nest_library_1.logger.info({
                webhookId,
                partnerId: webhook.partnerId,
                retryCount: webhook.retryCount,
            }, 'Webhook resend triggered');
            return {
                webhookId,
                status: webhook_enum_1.EWebhookStatus.pending,
                retryCount: webhook.retryCount,
                message: 'Webhook resend triggered successfully',
            };
        }
        catch (error) {
            la_nest_library_1.logger.error({
                webhookId,
                error: error.message,
            }, 'Failed to resend webhook');
            throw error;
        }
    }
    async getWebhookById(webhookId, auth) {
        const webhook = await this.webhookRepository.findOne({
            where: { id: webhookId },
            relations: ['partner', 'smsTransaction'],
        });
        if (!webhook) {
            throw new Error('Webhook not found');
        }
        if (auth.isClient && webhook.partnerId !== auth.id) {
            throw new Error('Unauthorized to view this webhook');
        }
        return webhook;
    }
};
exports.WebhookService = WebhookService;
exports.WebhookService = WebhookService = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [api_service_1.APIService,
        webhook_repository_1.WebhookRepository,
        sms_repository_1.SMSRepository,
        sms_transaction_repository_1.SMSTransactionRepository,
        partner_repository_1.PartnerRepository])
], WebhookService);
let WebhookServiceModule = class WebhookServiceModule {
};
exports.WebhookServiceModule = WebhookServiceModule;
exports.WebhookServiceModule = WebhookServiceModule = __decorate([
    (0, common_1.Global)(),
    (0, common_1.Module)({
        imports: [typeorm_1.TypeOrmModule.forFeature([webhook_entity_1.WebhookEntity])],
        providers: [WebhookService, webhook_repository_1.WebhookRepository],
        exports: [WebhookService],
    })
], WebhookServiceModule);
//# sourceMappingURL=webhook.service.js.map