/**
 * Utility type that extracts keys from T where the value is of type TExpectedFieldType
 * @template T Source object type
 * @template TExpectedFieldType Expected type of the field values to include
 * @remarks
 * This type utility filters object properties to only those matching a specific type.
 * It's useful for type-safe property access and filtering object keys based on their value types.
 *
 * @example
 * ```ts
 * interface User {
 *   name: string;
 *   age: number;
 *   email: string;
 *   isActive: boolean;
 * }
 *
 * // Will only include 'name' and 'email' keys
 * type StringFields = KeyOfType<User, string>;
 *
 * // Will only include 'age' key
 * type NumberFields = KeyOfType<User, number>;
 * ```
 */
export type KeyOfType<T, TExpectedFieldType> = keyof {
    [P in keyof T as T[P] extends TExpectedFieldType ? P : never]: any;
};
