"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MailService = void 0;
const nodemailer = require("nodemailer");
const logger_service_1 = require("./logger.service");
const index_config_1 = require("../configs/index.config");
class MailService {
    static loadConfig(config) {
        if (config.smtp) {
            index_config_1.SDKConfig.email = config.smtp;
            this.baseEmailSuffix = '@' + config.smtp.address?.split('@')[1];
        }
        if (config.adminNotificationMails)
            index_config_1.SDKConfig.adminNotificationMails = config.adminNotificationMails;
    }
    static loadConfigFromEnvironment(environment) {
        return this.loadConfig({
            smtp: {
                server: environment.SMTP_HOST,
                smtpPort: environment.SMTP_PORT,
                address: environment.SYSTEM_EMAIL_ADDRESS,
                password: environment.SYSTEM_EMAIL_PASSWORD,
            },
        });
    }
    static sendMail({ throwErr, ...config }) {
        const transporter = nodemailer.createTransport({
            host: index_config_1.SDKConfig.email.server,
            port: index_config_1.SDKConfig.email.smtpPort,
            secure: true, // upgrade later with STARTTLS
            auth: {
                user: index_config_1.SDKConfig.email.address,
                pass: index_config_1.SDKConfig.email.password,
            },
        });
        config.from =
            config.from ||
                `${config._senderName || index_config_1.SDKConfig.appName} <${index_config_1.SDKConfig.email.address}>`;
        config.sender = config.sender || config.from;
        config.subject =
            config.subject || `New Mail From ${index_config_1.SDKConfig.appName || config.sender}`;
        // verify connection configuration
        return transporter
            .verify()
            .then((r) => {
            console.log(r);
            // logger.info('Server is ready to send messages');
            return transporter
                .sendMail(config)
                .then((r) => {
                logger_service_1.logger.info({ message: 'Sent mail', response: r });
                return r;
            })
                .catch((e) => {
                logger_service_1.logger.error(e, 'Send Mail');
                if (throwErr)
                    throw e;
            });
        })
            .catch((error) => {
            logger_service_1.logger.error(error, 'Send Mail');
            if (throwErr)
                throw error;
        });
    }
    static sendCrashNotification(error) {
        return this.sendToAdmin({
            subject: 'Server Crashed',
            text: `Server crashed\nTime: ${new Date()}\n\n\nError\n\n${error?.toString()}`,
        });
    }
    static sendToAdmin(data) {
        return this.sendMail({
            to: index_config_1.SDKConfig.adminNotificationMails,
            from: index_config_1.SDKConfig.email.address,
            subject: data.subject,
            text: data.text,
            html: data.html,
        });
    }
}
exports.MailService = MailService;
//# sourceMappingURL=mail.service.js.map