"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var FileService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileService = exports.FileRoutePath = void 0;
const common_1 = require("@nestjs/common");
const file_entity_1 = require("../entities/file.entity");
const typeorm_1 = require("typeorm");
const utility_service_1 = require("./utility.service");
const typeorm_2 = require("@nestjs/typeorm");
const search_service_1 = require("./search.service");
const base_enum_1 = require("../enums/base.enum");
const likes_service_1 = require("./likes.service");
const search_enum_1 = require("../enums/search.enum");
const token_service_1 = require("./token.service");
const route_path_class_1 = require("../classes/route-path.class");
const rxjs_1 = require("rxjs");
const refCat = base_enum_1.ERefCat.file;
exports.FileRoutePath = new route_path_class_1.RoutePath('files');
let FileService = FileService_1 = class FileService {
    constructor(fileRepository, fileContentRepository, likesService, tokenService) {
        this.fileRepository = fileRepository;
        this.fileContentRepository = fileContentRepository;
        this.likesService = likesService;
        this.tokenService = tokenService;
        this.routePath = exports.FileRoutePath;
        this.maxUploadSize = 52_428_800 * 2;
        this.filesToRegenerate = new Map();
        this.regenerationTrigger$ = new rxjs_1.Subject();
        this.b64toBlob = (b64Data, contentType = '', sliceSize = 512) => {
            const byteCharacters = atob(b64Data);
            const byteArrays = [];
            for (let offset = 0; offset < byteCharacters.length; offset += sliceSize) {
                const slice = byteCharacters.slice(offset, offset + sliceSize);
                const byteNumbers = new Array(slice.length);
                for (let i = 0; i < slice.length; i++) {
                    byteNumbers[i] = slice.charCodeAt(i);
                }
                const byteArray = new Uint8Array(byteNumbers);
                byteArrays.push(byteArray);
            }
            const blob = new Blob(byteArrays, { type: contentType });
            return blob;
        };
        this.regenerationTrigger$
            .pipe((0, rxjs_1.tap)((file) => file ? this.filesToRegenerate.set(file.id, file) : null), (0, rxjs_1.debounceTime)(1000), (0, rxjs_1.filter)(() => this.filesToRegenerate.size > 0), (0, rxjs_1.exhaustMap)(() => this.regenerateAllLinks()))
            .subscribe();
    }
    async upload(multerFile, meta) {
        const nameChunks = multerFile.originalname.split('.');
        return this._upload({
            // blob: new Blob([multerFile.buffer]),
            fileSize: multerFile.size,
            mimetype: multerFile.mimetype,
            path: multerFile.path,
            fileExtension: nameChunks[nameChunks.length - 1],
        }, meta);
    }
    async _upload(file, meta) {
        const saveContentToDB = file.saveContentToDB || !file.path;
        if (file.fileSize > this.maxUploadSize)
            utility_service_1.UtilityClass.throwError({
                message: `File size ${this.formatFileSize(file.fileSize)} is larger than ${this.formatFileSize(this.maxUploadSize)}`,
            });
        const dbItem = this.fileRepository.create(meta);
        dbItem.fileSize = file.fileSize;
        dbItem.fileType = file.mimetype;
        dbItem.path = file.path;
        dbItem.title = dbItem.title;
        dbItem.fileExtension = file.fileExtension;
        // debugger;
        if (saveContentToDB)
            dbItem.content = this.fileContentRepository.create({
                dataURI: file.dataURI,
            });
        const existing = await this.fileRepository.findOne({
            where: { refCat: dbItem.refCat, refNo: dbItem.refNo },
            select: { id: true, contentId: true },
        });
        if (existing?.contentId)
            dbItem.content = {
                dataURI: null,
                ...dbItem.content,
                id: existing.contentId,
            };
        // if (!dbItem.creator?.id) delete dbItem.creator;
        // if (!dbItem.creator?.id) delete dbItem.creator;
        const res = await (existing
            ? this.fileRepository.save({ ...dbItem, id: existing.id })
            : this.fileRepository.save(dbItem));
        // delete dbItem.path;
        return this.getFileForPublic({
            id: existing?.id || res.id,
        });
    }
    uploadJSON({ fileExtension, dataURI, ...meta }) {
        const chunks = dataURI.split(';base64,'), mimetype = chunks[0].slice(5);
        return this._upload({
            // dataURI,
            dataURI: chunks[1],
            fileExtension,
            mimetype,
            saveContentToDB: true,
            fileSize: atob(chunks[1]).length,
        }, meta);
    }
    base64ToArrayBuffer(base64) {
        const binaryString = atob(base64);
        const bytes = new Uint8Array(binaryString.length);
        for (let i = 0; i < binaryString.length; i++) {
            bytes[i] = binaryString.charCodeAt(i);
        }
        return bytes.buffer;
    }
    async updateFile(body) {
        await this.checkIfFileExists(body.fileID);
        await this.fileRepository.update({ id: body.fileID }, body);
    }
    async likeFile(params) {
        const res = await this.likesService.likeFile({
            refID: params.fileID,
            refCat: refCat,
            createdBy: params.creatorId,
        });
        return await this.updateLikesDislikes(params.fileID, res);
    }
    async dislikeFile(params) {
        const res = await this.likesService.dislikeFile({
            refID: params.fileID,
            refCat: refCat,
            createdBy: params.creatorId,
        });
        return await this.updateLikesDislikes(params.fileID, res);
    }
    async unlikeFile(params) {
        const res = await this.likesService.unlikeFile({
            refID: params.fileID,
            refCat: refCat,
            createdBy: params.creatorId,
        });
        return await this.updateLikesDislikes(params.fileID, res);
    }
    async updateLikesDislikes(albumID, data) {
        if (!data)
            data = await this.likesService.calculateLikesDislikes(albumID);
        await this.fileRepository.update({ id: albumID }, { likes: data.likes, dislikes: data.dislikes });
        return this.fileRepository.findOne({
            where: { id: albumID },
            select: {
                id: true,
                likes: true,
                dislikes: true,
            },
        });
    }
    async deleteSingleFile(params) {
        return await this.fileRepository.softDelete({ id: params.fileID });
    }
    async deleteFileBatch(params) {
        return this.fileRepository.softDelete({ id: (0, typeorm_1.In)(params.ids) });
    }
    async restoreFileSingle(params) {
        await this.fileRepository.restore({ id: params.fileID });
        return this.getFileByID(params.fileID);
    }
    async restoreFileBatch(params) {
        return this.fileRepository.restore({ id: (0, typeorm_1.In)(params.ids) });
    }
    async getFileByID(fileID) {
        const existing = await this.fileRepository.findOne({
            where: {
                id: fileID,
            },
        });
        if (!existing)
            utility_service_1.UtilityClass.throwError({
                message: `File does not exist`,
                statusCode: 404,
            });
        return existing;
    }
    async checkIfFileExists(fileID) {
        const existing = await this.fileRepository.count({
            where: {
                id: fileID,
            },
        });
        if (existing !== 1)
            utility_service_1.UtilityClass.throwError({
                message: `File does not exist`,
                statusCode: 404,
            });
        return true;
    }
    async search(query) {
        const res = await search_service_1.SearchService.search({
            repository: this.fileRepository,
            tableName: base_enum_1.ETableName.file,
        }, query, [
            { field: 'description', condition: search_enum_1.ESearchCondition.contains },
            { field: 'title', condition: search_enum_1.ESearchCondition.contains },
            { field: 'refCat' },
            { field: 'refNo' },
        ]);
        return res;
    }
    async getFileForPublic(query) {
        const file = await this.fileRepository.findOne({ where: query });
        if (!file)
            return null;
        if (!file.link)
            file.link = await this.generateFileLink(file);
        return file;
    }
    async regenerateAllLinks() {
        let files;
        const leftFiles = await this.fileRepository.findBy({ link: (0, typeorm_1.IsNull)() });
        leftFiles.forEach((l) => {
            this.filesToRegenerate.set(l.id, l);
        });
        files = Array.from(this.filesToRegenerate.values());
        if (files?.length)
            this.filesToRegenerate.clear();
        else {
            await this.fileRepository.update({ link: (0, typeorm_1.Not)((0, typeorm_1.IsNull)()) }, { link: null });
            files = await this.fileRepository.find();
        }
        return (0, rxjs_1.lastValueFrom)((0, rxjs_1.concat)(files.map((f) => this.generateFileLink(f))));
    }
    async generateFileLink(param) {
        const file = typeof param == 'string'
            ? await this.fileRepository.findOneBy({ id: param })
            : param;
        if (!file)
            utility_service_1.UtilityClass.throwError({ statusCode: 404, message: 'File not found' });
        const link = `${utility_service_1.UtilityClass.getHost}${this.routePath.path}/${FileService_1.downloadPathGeneratorWithToken(await this.tokenizeFile(file))}`;
        await this.fileRepository.update({ id: file.id }, { link });
        return link;
    }
    tokenizeFile(params) {
        return this.tokenService
            .generateToken({
            title: params.title,
            fileExtension: params.fileExtension,
            fileType: params.fileType,
            contentId: params.contentId,
            path: params.path,
            id: params.id,
        }, {
            expiresIn: '90d',
        })
            .then((r) => r.token);
    }
    async getFilePath(params) {
        const file = params.contentId || params.path
            ? params
            : await this.fileRepository.findOne({
                where: { id: params.fileID || params.id },
                select: {
                    path: true,
                    fileExtension: true,
                    fileType: true,
                    title: true,
                    contentId: true,
                },
            });
        if (!file)
            throw new common_1.HttpException('File could not be found', 404);
        return {
            path: file.path ? `./${file.path}` : null,
            file,
            buffer: file.contentId
                ? (await this.fileContentRepository.findOne({
                    where: {
                        id: file.contentId,
                    },
                }))?.dataURI
                : null,
        };
    }
    async getFilePathWithToken(params) {
        // debugger;
        let fileToken;
        try {
            fileToken = await this.tokenService.decryptToken(params.token);
        }
        catch (error) {
            try {
                fileToken = await this.tokenService.decryptToken(params.token, {
                    ignoreExpiration: true,
                });
                this.generateFileLink(fileToken.id || fileToken.fileID);
            }
            catch (error) {
                fileToken = null;
            }
        }
        if (!fileToken)
            utility_service_1.UtilityClass.throwError({ statusCode: 404, message: `File not found` });
        return this.getFilePath(fileToken);
    }
    /**
     * Format bytes as human-readable text.
     *
     * @param bytes Number of bytes.
     * @param si True to use metric (SI) units, aka powers of 1000. False to use
     *           binary (IEC), aka powers of 1024.
     * @param dp Number of decimal places to display.
     *
     * @return Formatted string.
     */
    formatFileSize(bytes, si = false, dp = 1) {
        const thresh = si ? 1000 : 1024;
        if (Math.abs(bytes) < thresh) {
            return bytes + ' B';
        }
        const units = si
            ? ['kB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB']
            : ['KiB', 'MiB', 'GiB', 'TiB', 'PiB', 'EiB', 'ZiB', 'YiB'];
        let u = -1;
        const r = 10 ** dp;
        do {
            bytes /= thresh;
            ++u;
        } while (Math.round(Math.abs(bytes) * r) / r >= thresh &&
            u < units.length - 1);
        return bytes.toFixed(dp) + ' ' + units[u];
    }
};
exports.FileService = FileService;
FileService.downloadPathGenerator = (fileID) => `getFile/${fileID}`;
FileService.downloadPathGeneratorWithToken = (token) => `getFile/token/${token}`;
exports.FileService = FileService = FileService_1 = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_2.InjectRepository)(file_entity_1.FileEntity)),
    __param(1, (0, typeorm_2.InjectRepository)(file_entity_1.FileContentEntity)),
    __metadata("design:paramtypes", [typeorm_1.Repository,
        typeorm_1.Repository,
        likes_service_1.LikesService,
        token_service_1.TokenService])
], FileService);
//# sourceMappingURL=file.service.js.map