"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var AuthenticationGuard_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthParam = exports.QueryParamAuth = exports.PublicAuth = exports.AuthenticationGuard = void 0;
const common_1 = require("@nestjs/common");
const token_service_1 = require("../../services/token.service");
const authentication_service_1 = require("./authentication.service");
const authentication_enum_1 = require("./authentication.enum");
const common_2 = require("@nestjs/common");
const core_1 = require("@nestjs/core");
const environment_service_1 = require("../../services/environment.service");
/**
 * Authentication guard that verifies bearer tokens in incoming HTTP requests.
 *
 * @typeParam TEntity - Concrete `AuthEntity` used by the authentication service.
 *
 * @example
 * ```ts
 * @UseGuards(AuthenticationGuard)
 * @Controller('account')
 * export class AccountController {
 *   @Get()
 *   getProfile(@AuthParam() auth: IAuthParam) {
 *     return auth;
 *   }
 * }
 * ```
 */
let AuthenticationGuard = AuthenticationGuard_1 = class AuthenticationGuard {
    constructor(tokenService, authS, reflector) {
        this.tokenService = tokenService;
        this.authS = authS;
        this.reflector = reflector;
        /**
         * Environment identifier at which authentication checks are skipped.
         *
         * @example
         * ```ts
         * guard.skipEnvironment = 'test';
         * ```
         */
        this.skipEnvironment = 'dev';
    }
    /**
     * Verifies the provided authentication token.
     *
     * @param context - Current execution context.
     * @param payload - Payload containing the bearer token.
     * @returns Result of the authentication service verification.
     *
     * @example
     * ```ts
     * const isValid = await guard.verifyAuthToken(context, payload);
     * ```
     */
    verifyAuthToken(context, payload) {
        return this.authS.verifyAuthToken(payload.token);
    }
    /**
     * Determines whether a request is allowed to proceed.
     *
     * @param context - Execution context containing the request.
     * @returns Resolves with `true` when the request is authorized.
     *
     * @example
     * ```ts
     * const allowed = await guard.canActivate(context);
     * ```
     */
    async canActivate(context) {
        const isPublic = this.reflector.getAllAndOverride(IS_PUBLIC_KEY, [
            context.getHandler(),
            context.getClass(),
        ]);
        const isQueryParamAuth = this.reflector.getAllAndOverride(IS_QUERY_PARAM_AUTH_KEY, [context.getHandler(), context.getClass()]);
        // debugger;
        const request = context.switchToHttp().getRequest();
        const token = isQueryParamAuth
            ? this.extractTokenFromQueryParam(request)
            : this.extractTokenFromHeader(request);
        const skip = environment_service_1.evt.ENVIRONMENT == this.skipEnvironment;
        // const skip = true;
        // const d = evt.ENVIRONMENT;
        if (!token && !skip) {
            if (isPublic)
                return true;
            throw new common_1.UnauthorizedException();
        }
        let payload;
        try {
            if (token) {
                payload = await this.tokenService.decryptToken(token, {
                    ignoreExpiration: skip,
                });
                payload.token = token;
                await this.verifyAuthToken(context, payload);
                await this.requestPatcher(request, payload);
            }
        }
        catch (error) {
            this.authS.forceLogoutSingle({
                token,
                dropReason: authentication_enum_1.ETokenDropType.expiration,
            });
            if (isPublic)
                return true;
            throw new common_1.UnauthorizedException('Token Expired');
        }
        return this.extraCheck
            ? await this.extraCheck(context, payload, isPublic)
            : true;
    }
    /**
     * Extracts the bearer token from the HTTP authorization header.
     *
     * @param request - Incoming HTTP request.
     * @returns The bearer token if present; otherwise `undefined`.
     *
     * @example
     * ```ts
     * const token = guard['extractTokenFromHeader'](request);
     * ```
     */
    extractTokenFromHeader(request) {
        return this.extractTokenFromTokenString(request.headers.authorization);
    }
    extractTokenFromQueryParam(request) {
        const authorization = request.query.authorization;
        return typeof authorization != 'string'
            ? undefined
            : this.extractTokenFromTokenString(authorization);
    }
    extractTokenFromTokenString(tokenString) {
        const [type, token] = tokenString?.split(' ') ?? [];
        return type === 'Bearer' ? token : undefined;
    }
    /**
     * Attaches the authentication payload to the request object.
     *
     * @param request - HTTP request being processed.
     * @param payload - Decrypted authentication payload.
     *
     * @example
     * ```ts
     * await guard.requestPatcher(req, payload);
     * console.log(req.user);
     * ```
     */
    async requestPatcher(request, payload) {
        request['user'] = payload || null;
    }
    /**
     * Provides the guard as an application-wide provider.
     *
     * @returns Provider definition for the guard.
     *
     * @example
     * ```ts
     * @Module({
     *   providers: [AuthenticationGuard.provideGuard()],
     * })
     * export class AppModule {}
     * ```
     */
    static provideGuard() {
        return {
            provide: core_1.APP_GUARD,
            useClass: AuthenticationGuard_1,
        };
    }
};
exports.AuthenticationGuard = AuthenticationGuard;
exports.AuthenticationGuard = AuthenticationGuard = AuthenticationGuard_1 = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [token_service_1.TokenService,
        authentication_service_1.AuthenticationService,
        core_1.Reflector])
], AuthenticationGuard);
const IS_PUBLIC_KEY = 'isPublic';
/**
 * Marks a route handler as public, bypassing authentication.
 *
 * @returns Metadata decorator indicating the handler is public.
 *
 * @example
 * ```ts
 * @PublicAuth()
 * @Get('status')
 * status() {
 *   return 'ok';
 * }
 * ```
 */
const PublicAuth = () => (0, common_2.SetMetadata)(IS_PUBLIC_KEY, true);
exports.PublicAuth = PublicAuth;
const IS_QUERY_PARAM_AUTH_KEY = 'IS_QUERY_PARAM_AUTH';
const QueryParamAuth = () => (0, common_2.SetMetadata)(IS_QUERY_PARAM_AUTH_KEY, true);
exports.QueryParamAuth = QueryParamAuth;
/**
 * Retrieves the authentication payload from the request.
 *
 * @returns Extracted authentication parameter.
 *
 * @example
 * ```ts
 * someHandler(@AuthParam() auth) {
 *   console.log(auth.id);
 * }
 * ```
 */
exports.AuthParam = (0, common_1.createParamDecorator)((data, req) => req.getArgByIndex(0).user);
//# sourceMappingURL=authentication.guard.js.map