import { CanActivate, ExecutionContext } from '@nestjs/common';
import { TokenService } from '../../services/token.service';
import { AuthenticationService } from './authentication.service';
import { Reflector } from '@nestjs/core';
import { evt } from '../../services/environment.service';
import { IAuthParam } from './authentication.interface';
import { AuthEntity } from './authentication.entity';
/**
 * Authentication guard that verifies bearer tokens in incoming HTTP requests.
 *
 * @typeParam TEntity - Concrete `AuthEntity` used by the authentication service.
 *
 * @example
 * ```ts
 * @UseGuards(AuthenticationGuard)
 * @Controller('account')
 * export class AccountController {
 *   @Get()
 *   getProfile(@AuthParam() auth: IAuthParam) {
 *     return auth;
 *   }
 * }
 * ```
 */
export declare class AuthenticationGuard<TEntity extends AuthEntity = AuthEntity> implements CanActivate {
    protected tokenService: TokenService;
    protected authS: AuthenticationService<TEntity>;
    protected reflector: Reflector;
    /**
     * Environment identifier at which authentication checks are skipped.
     *
     * @example
     * ```ts
     * guard.skipEnvironment = 'test';
     * ```
     */
    skipEnvironment: typeof evt.ENVIRONMENT;
    constructor(tokenService: TokenService, authS: AuthenticationService<TEntity>, reflector: Reflector);
    /**
     * Optional additional validation executed after the token is verified.
     *
     * @param context - Current execution context.
     * @param payload - Decrypted authentication payload.
     * @param isPublic - Whether the current handler is marked as public.
     *
     * @example
     * ```ts
     * guard.extraCheck = async (context, payload) => payload.role === 'admin';
     * ```
     */
    extraCheck?: (context: ExecutionContext, payload: IAuthParam, isPublic: boolean) => Promise<boolean>;
    /**
     * Verifies the provided authentication token.
     *
     * @param context - Current execution context.
     * @param payload - Payload containing the bearer token.
     * @returns Result of the authentication service verification.
     *
     * @example
     * ```ts
     * const isValid = await guard.verifyAuthToken(context, payload);
     * ```
     */
    verifyAuthToken(context: ExecutionContext, payload: IAuthParam): Promise<void>;
    /**
     * Determines whether a request is allowed to proceed.
     *
     * @param context - Execution context containing the request.
     * @returns Resolves with `true` when the request is authorized.
     *
     * @example
     * ```ts
     * const allowed = await guard.canActivate(context);
     * ```
     */
    canActivate(context: ExecutionContext): Promise<boolean>;
    /**
     * Extracts the bearer token from the HTTP authorization header.
     *
     * @param request - Incoming HTTP request.
     * @returns The bearer token if present; otherwise `undefined`.
     *
     * @example
     * ```ts
     * const token = guard['extractTokenFromHeader'](request);
     * ```
     */
    private extractTokenFromHeader;
    private extractTokenFromQueryParam;
    private extractTokenFromTokenString;
    /**
     * Attaches the authentication payload to the request object.
     *
     * @param request - HTTP request being processed.
     * @param payload - Decrypted authentication payload.
     *
     * @example
     * ```ts
     * await guard.requestPatcher(req, payload);
     * console.log(req.user);
     * ```
     */
    requestPatcher(request: any, payload: any): Promise<void>;
    /**
     * Provides the guard as an application-wide provider.
     *
     * @returns Provider definition for the guard.
     *
     * @example
     * ```ts
     * @Module({
     *   providers: [AuthenticationGuard.provideGuard()],
     * })
     * export class AppModule {}
     * ```
     */
    static provideGuard(): {
        provide: string;
        useClass: typeof AuthenticationGuard;
    };
}
/**
 * Marks a route handler as public, bypassing authentication.
 *
 * @returns Metadata decorator indicating the handler is public.
 *
 * @example
 * ```ts
 * @PublicAuth()
 * @Get('status')
 * status() {
 *   return 'ok';
 * }
 * ```
 */
export declare const PublicAuth: () => import("@nestjs/common").CustomDecorator<string>;
export declare const QueryParamAuth: () => import("@nestjs/common").CustomDecorator<string>;
/**
 * Retrieves the authentication payload from the request.
 *
 * @returns Extracted authentication parameter.
 *
 * @example
 * ```ts
 * someHandler(@AuthParam() auth) {
 *   console.log(auth.id);
 * }
 * ```
 */
export declare const AuthParam: (...dataOrPipes: any[]) => ParameterDecorator;
