"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphPublisher = void 0;
const serialized_graph_1 = require("@nestjs/core/inspector/serialized-graph");
const chalk_1 = __importDefault(require("chalk"));
const node_fetch_1 = __importDefault(require("node-fetch"));
const util_1 = require("util");
const zlib_1 = require("zlib");
const promisifiedDeflate = (0, util_1.promisify)(zlib_1.deflate);
class GraphPublisher {
    constructor(app) {
        this.serializedGraph = app.get(serialized_graph_1.SerializedGraph);
    }
    publish(options) {
        var _a, _b;
        return __awaiter(this, void 0, void 0, function* () {
            this.assertOption(options, 'owner');
            this.assertOption(options, 'repository');
            this.assertOption(options, 'sha');
            this.assertOption(options, 'branch');
            const organizationSlug = options.owner;
            const projectSlug = options.repository;
            const devtoolsUrl = (_a = process.env.DEVTOOLS_API_URL) !== null && _a !== void 0 ? _a : 'https://devtools-api.nestjs.com';
            const url = `${devtoolsUrl}/organizations/${organizationSlug}/projects/${projectSlug}/builds`;
            const snapshot = this.serializedGraph.toString();
            const deflatedSnapshot = yield promisifiedDeflate(snapshot);
            const response = yield (0, node_fetch_1.default)(url, {
                method: 'POST',
                headers: {
                    Authorization: `ApiKey ${options.apiKey}`,
                    Accept: 'application/json',
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    branch: options.branch,
                    trigger: (_b = options.trigger) !== null && _b !== void 0 ? _b : 'push',
                    snapshot: deflatedSnapshot.toString('base64'),
                    hash: options.sha,
                    target: options.target || undefined,
                }),
            });
            if (response.ok) {
                const data = yield response.json();
                this.printGraphPublishedLog(data);
            }
            else {
                if (response.status === 401) {
                    this.printUnauthorizedErrorLog();
                }
                else if (response.status === 400) {
                    this.printBadRequestErrorLog();
                }
                else {
                    this.printUnknownErrorLog();
                }
                console.error('Error:', yield response.text(), response.status);
            }
        });
    }
    printGraphPublishedLog(data) {
        var _a;
        console.log((0, chalk_1.default) `
{bgGreen {black  DEVTOOLS }} {green Graph has been successfully {bold published}.}

{cyan -} Graph preview URL: {underline {cyan ${data.previewUrl}}}
{cyan -} Report URL: {underline {cyan ${(_a = data.reportUrl) !== null && _a !== void 0 ? _a : '-'}}}

{grey ${data.reportUrl
            ? 'NOTE: This may take some time before the report is available.'
            : ''}}
`);
    }
    assertOption(options, key) {
        if (!options[key]) {
            throw new Error(`GraphPublisher: "${key}" parameter is required`);
        }
    }
    printUnauthorizedErrorLog() {
        console.log((0, chalk_1.default) `
{bgRed {black  DEVTOOLS }} {red Error! Graph has not been successfully published.}

{red -} API returned {red {bold 401}} status code. Please check your API key.
`);
    }
    printUnknownErrorLog() {
        console.log((0, chalk_1.default) `
{bgRed {black  DEVTOOLS }} {red Error! Something went wrong.}

`);
    }
    printBadRequestErrorLog() {
        console.log((0, chalk_1.default) `
{bgRed {black  DEVTOOLS }} {red Error! Graph has not been successfully published.}

{red -} API returned {red {bold 400}} status code. Please double-check if the payload is valid.
`);
    }
}
exports.GraphPublisher = GraphPublisher;
